unit fROR_GenericRegistry;

interface

uses
  Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls, Forms,
  Dialogs, fROR_MDIChild, fROR_VistARegistries, ExtCtrls, ComCtrls, Menus,
  ActnList, ImgList, Buttons, ToolWin, StdCtrls, uROR_PatientData, uROR_Reports,
  uROR_Common, ovcfiler, ovcstore, ovcbase, ovcstate, uROR_Errors,
  uROR_ListView, uROR_VistAStore, uROR_Selector, uROR_CustomControls,
  uROR_Contextor, uROR_CustomListView, uROR_GridView, HTMLLite, uROR_Logs,
  Grids, AdvGrid, AdvCGrid, uROR_SearchEdit, AdvMenus, AdvMenuStylers,
  VA508AccessibilityManager, uROR_DebugBox;

const
  iPtlDummy             =  0;
  iPtlName              =  1;
  iPtlDOB               =  2;
  iPtlSSN               =  3;
  iPtlConfirmed         =  4;
  iPtlStatus            =  5;
  iPtlSite              =  6;
  iPtlSelected          =  7;
  iPtlSelectionRule     =  8;
  iPtlFlags             =  9;
  iPtlIEN               = 10;
  iPtlDOD               = 11;
  iPtlSex               = 12;
  iPtlExtracted         = 13;
  iPtlAcknowledged      = 14;
  iPtlStatusCode        = 15;
  iPtlComment           = 16;

  iTskDummy             =  0;
  iTskNumber            =  1;
  iTskType              =  2;
  iTskDescription       =  3;
  iTskScheduled         =  4;
  iTskStatus            =  5;
  iTskProgress          =  6;
  iTskCompleted         =  7;
  iTskComment           =  8;
  iTskReportCode        =  9;
  iTskLogIEN            = 10;

type
  TFormGenericRegistry = class(TFormMDIChild)
    pnlClientArea: TPanel;
    PageControl: TPageControl;
    tsTaskManager: TTabSheet;
    tsActivityLog: TTabSheet;
    tsRegistry: TTabSheet;
    mnuTaskList: TAdvPopupMenu;
    NewReport1: TMenuItem;
    mnuOpenReport: TMenuItem;
    mnuRescheduleTask: TMenuItem;
    mnuDeleteTask: TMenuItem;
    mnuN3: TMenuItem;
    mnuRefreshTaskList: TMenuItem;
    aclActionList: TActionList;
    acRescheduleTask: TAction;
    acDeleteTask: TAction;
    acRefreshTaskList: TAction;
    acOpenReport: TAction;
    acRefreshActivityLog: TAction;
    acAddTask: TAction;
    acShowRegistryUsers: TAction;
    acHelpTopics: TAction;
    acHelpRegistryInfo: TAction;
    acHelpAbout: TAction;
    acSiteParameters: TAction;
    acEditPatientData: TAction;
    acDeletePatient: TAction;
    mnuGenericMain: TAdvMainMenu;
    mnuRegistry: TMenuItem;
    mnuAddEditPatient: TMenuItem;
    N1: TMenuItem;
    mnuShowRegistryUsers: TMenuItem;
    mnuEditSiteParams: TMenuItem;
    mnuGenericReports: TMenuItem;
    mnuHelp: TMenuItem;
    mnuHelpTopics: TMenuItem;
    N2: TMenuItem;
    mnuHelpRegInfo: TMenuItem;
    N3: TMenuItem;
    mnuHelpAbout: TMenuItem;
    pnlTasks: TPanel;
    pnlActivityLog: TPanel;
    Splitter1: TSplitter;
    pnlTree: TPanel;
    pnlDetails: TPanel;
    pnlRegistry: TPanel;
    pnlSearch: TPanel;
    cbScreen: TCheckBox;
    dtpScreenDate: TDateTimePicker;
    cbPending: TCheckBox;
    ImageList: TImageList;
    lsvTaskList: TCCRGridView;
    lsvPatientList: TCCRGridView;
    vxsFormLayout: TCCRVistAStore;
    acHelpCCOWStatus: TAction;
    mnuHelpCCOWStatus: TMenuItem;
    ImageListPatients: TImageList;
    tvLogs: TTreeView;
    htmlDetails: ThtmlLite;
    acViewTaskLog: TAction;
    mnuViewTaskLog: TMenuItem;
    mnuLogs: TAdvPopupMenu;
    acMsgSelectAll: TAction;
    acMsgCopy: TAction;
    mnuLogsCopy: TMenuItem;
    mnuLogsSelectAll: TMenuItem;
    stxtCount: TStaticText;
    csePatient: TCCRVistASearchEdit;
    acCancelSearch: TAction;
    pnlToolbarTasks: TPanel;
    btnRefresh: TBitBtn;
    btnNewReport: TBitBtn;
    btnOpenReport: TBitBtn;
    btnViewTaskLog: TBitBtn;
    btnDeleteTask: TBitBtn;
    btnTasksSpacer: TBitBtn;
    pnlToolbarLogs: TPanel;
    btnRefreshTechnicalLog: TBitBtn;
    pnlLogsSpacer: TPanel;
    pnlLogsFromTo: TPanel;
    dtpFrom: TDateTimePicker;
    dtpTo: TDateTimePicker;
    pnlToolbarRegistry: TPanel;
    btnPatientSearch: TBitBtn;
    btnPatientEdit: TBitBtn;
    btnPatientDelete: TBitBtn;
    btnRegistrySpacer: TBitBtn;
    amsMenuStyler: TAdvMenuOfficeStyler;
    pnlDetails508: TPanel;
    stxtTitle: TStaticText;
    stxtLabel1: TStaticText;
    stxtLabel2: TStaticText;
    stxtLabel3: TStaticText;
    stxtLabel4: TStaticText;
    stxtLabel5: TStaticText;
    stxtLabel6: TStaticText;
    stxtLabel7: TStaticText;
    stxtLabel8: TStaticText;
    stxtLogDateFrom: TStaticText;
    stxtLogDateTo: TStaticText;
    stxtPatientSearch: TStaticText;
    mnuTechLog: TAdvPopupMenu;
    MenuItem1: TMenuItem;
    procedure acHelpRegistryInfoExecute(Sender: TObject);
    procedure acHelpAboutExecute(Sender: TObject);
    procedure aclActionListUpdate(Action: TBasicAction;
      var Handled: Boolean);
    procedure acShowRegistryUsersExecute(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure mnuRegistryClick(Sender: TObject);
    procedure PageControlChange(Sender: TObject);
    procedure acAddTaskExecute(Sender: TObject);
    procedure acDeleteTaskExecute(Sender: TObject);
    procedure acRefreshTaskListExecute(Sender: TObject);
    procedure acOpenReportExecute(Sender: TObject);
    procedure acRefreshActivityLogExecute(Sender: TObject);
    procedure acDeletePatientExecute(Sender: TObject);
    procedure acEditPatientDataExecute(Sender: TObject);
    procedure cbClearScreen(Sender: TObject);
    procedure cbScreenClick(Sender: TObject);
    procedure mnuReportsClick(Sender: TObject);
    procedure lsvTaskListDblClick(Sender: TObject);
    procedure lsvPatientListDblClick(Sender: TObject);
    procedure vxsFormLayoutBeforeOpen(Sender: TObject);
    procedure acHelpCCOWStatusExecute(Sender: TObject);
    procedure lsvPatientListCustomDrawItem(Sender: TCustomListView;
      Item: TListItem; State: TCustomDrawState; var DefaultDraw: Boolean);
    procedure lsvPatientListInsert(Sender: TObject; Item: TListItem);
    procedure tvLogsExpanding(Sender: TObject; Node: TTreeNode;
      var AllowExpansion: Boolean);
    procedure tvLogsChange(Sender: TObject; Node: TTreeNode);
    procedure tvLogsCollapsed(Sender: TObject; Node: TTreeNode);
    procedure tvLogsDeletion(Sender: TObject; Node: TTreeNode);
    procedure acViewTaskLogExecute(Sender: TObject);
    procedure acSiteParametersExecute(Sender: TObject);
    procedure csePatientSearchStart(Sender: TObject; var aCommand: TCCRSearchCommand);
    procedure mnuLogsPopup(Sender: TObject);
    procedure acMsgSelectAllExecute(Sender: TObject);
    procedure acMsgCopyExecute(Sender: TObject);
    procedure btnPatientSearchClick(Sender: TObject);
    procedure acCancelSearchExecute(Sender: TObject);
    procedure csePatientSearchEnd(aSender: TObject;
      var aCommand: TCCRSearchCommand);
    procedure csePatientSearch(aSender: TObject;
      var aCommand: TCCRSearchCommand);
    procedure pnlToolbarTasksResize(Sender: TObject);
    procedure pnlToolbarLogsResize(Sender: TObject);
    procedure pnlToolbarRegistryResize(Sender: TObject);
    procedure mnuGenericReportsClick(Sender: TObject);
    procedure lsvTaskListKeyPress(Sender: TObject; var Key: Char);
    procedure tvLogsKeyPress(Sender: TObject; var Key: Char);
    procedure lsvPatientListKeyDown(Sender: TObject; var Key: Word;
      Shift: TShiftState);
    procedure PageControlEnter(Sender: TObject);
    procedure FormActivate(Sender: TObject);
    procedure tsActivityLogEnter(Sender: TObject);
    procedure tsTaskManagerEnter(Sender: TObject);

  private
    fActivePatientData:     TPatientData;
    fHTMLParams:            TCCRHTMLParams;
    fPatientIEN:            String;
    fRegInfo:               TRegistryInfo;

    procedure LoadPatientList(aPatientIEN: String = '');

  protected
    PtIEN,
    PtName: String;

    function  AddLogNodes(RawData: TStrings): Integer;
    function  GetLayoutSectionName: String; virtual;
    procedure LoadLayout;
    procedure LoadLogMessages(aNode: TTreeNode);
    procedure ReloadActivePatientRecord(const aPatientIEN: String); virtual;
    procedure SaveLayout;
    procedure setActivePatientData(Data: TPatientData); virtual;
    function  SetPatientContext: Boolean; virtual;
    procedure SetupReportsMenu; virtual;
    procedure SwitchActivePatient(anItem: TCCRGridItem); virtual;
    procedure updatePatientCounter;
    procedure updateTaskList; virtual;

  public
    constructor Create(anOwner: TComponent); override;
    constructor CreateRegistry(aRegInfo: TRegistryInfo;
                  anOwner: TComponent = nil); virtual;
    destructor  Destroy; override;

    procedure Activity_LoadList; virtual;
    procedure CCOWResumed; override;
    procedure CCOWSuspended; override;
    procedure LoadTaskList;
    procedure Clear508LogDetails;
    procedure Show508LogDetails(const aLogDetail: String);
    function  OpenLog(const aLogIEN: String): Boolean;
    function  ScheduleReport(aReportCode: String = ''): TReturnCode; virtual;
    procedure SelectPatient(aPatientIEN: String); override;
    function  Tasks_Delete( ATask: String ): Integer; virtual;

    property ActivePatientData: TPatientData    read  fActivePatientData
                                                write setActivePatientData;

    property HTMLParams: TCCRHTMLParams         read  fHTMLParams;

    property RegInfo: TRegistryInfo             read  fRegInfo;

  published
    procedure ApplyOptions; override;

  end;

implementation
{$R *.dfm}

uses
  fROR_RegistryInfo, fROR_Options, fROR_UserList, uROR_Broker, fROR_Main,
  uROR_Utilities, uROR_CommonUtils, fROR_XMLReport, fROR_DateTimeSelector,
  uROR_Strings, VERGENCECONTEXTORLib_TLB, fROR_ReportSelector, fROR_SiteParams,
  uROR_Classes;

///////////////////////////// TFormGenericRegistry \\\\\\\\\\\\\\\\\\\\\\\\\\\\\

constructor TFormGenericRegistry.Create(anOwner: TComponent);
begin
  CreateRegistry(nil, anOwner);
end;

constructor TFormGenericRegistry.CreateRegistry(aRegInfo: TRegistryInfo;
  anOwner: TComponent);
begin
  if Assigned(anOwner) then
    inherited Create(anOwner)
  else
    inherited Create(Application);

  //-- Section 508: Set the tab stop on the patient count text so screen readers
  //   will recognize it
  if CCRScreenReaderActive then
  begin
    stxtCount.TabStop := true;
  end;

  fActivePatientData := nil;
  fPatientIEN := '';

  dtpTo.Date := Now;
  dtpFrom.Date := dtpTo.Date - iDefaultActivityTerm;

  fRegInfo := aRegInfo;
  if Assigned(fRegInfo) then
  begin
    Caption := fRegInfo.Description;
    fRegInfo.Load;

    fActivePatientData := TPatientData.Create(aRegInfo.IEN);
    SetupReportsMenu;

    if RegInfo.Admin then
    begin
      mnuAddEditPatient.Visible    := True;
      N1.Visible                   := True;
      mnuShowRegistryUsers.Visible := True;
      mnuEditSiteParams.Visible    := True;
      tsRegistry.TabVisible        := True;
      cbPending.Visible            := (RegInfo.AutoConfirm = false);
      cbPending.TabStop            := cbPending.Visible;
      lsvPatientList.Fields[iPtlComment].Visible := cbPending.Visible;
      PageControl.ActivePage       := tsRegistry;
    end
    else begin
      if PageControl.ActivePage = tsRegistry then
        PageControl.ActivePage := tsTaskManager;
      tsRegistry.TabVisible := False;
    end;

    SelectPatient(CCOWContextor.PatientDFN);
  end;

  fHTMLParams := TCCRHTMLParams.Create(htmlDetails);
end;

destructor TFormGenericRegistry.Destroy;
begin
  FreeAndNil(fHTMLParams);
  FreeAndNil(fActivePatientData);
  fRegInfo := nil;
  inherited Destroy;
end;

procedure TFormGenericRegistry.acCancelSearchExecute(Sender: TObject);
begin
  csePatient.CancelSearch;
end;

procedure TFormGenericRegistry.acHelpRegistryInfoExecute(Sender: TObject);
begin
  ShowInfo(RegInfo.IEN);
end;

procedure TFormGenericRegistry.acHelpAboutExecute(Sender: TObject);
begin
  TFormMain(Parent).acHelpAboutExecute(Sender);
end;

procedure TFormGenericRegistry.acHelpCCOWStatusExecute(Sender: TObject);
begin
  if CCOWContextor <> nil then
    CCOWContextor.DisplayStatus;
end;

procedure TFormGenericRegistry.aclActionListUpdate(Action: TBasicAction;
  var Handled: Boolean);
begin
  if lsvTaskList.SelCount > 0 then
  begin
    acDeleteTask.Enabled := True;
    acOpenReport.Enabled     := (lsvTaskList.SelCount = 1);
    acRescheduleTask.Enabled := (lsvTaskList.SelCount = 1);
    acViewTaskLog.Enabled    := (lsvTaskList.SelCount = 1);
  end
  else begin
    acDeleteTask.Enabled     := False;
    acOpenReport.Enabled     := False;
    acRescheduleTask.Enabled := False;
    acViewTaskLog.Enabled    := False;
  end;

  if lsvPatientList.Items.Count > 0 then
    SwitchActivePatient(lsvPatientList.ItemFocused)
  else
    SwitchActivePatient(nil);

  if Assigned(lsvPatientList.ItemFocused)
    and (lsvPatientList.ItemFocused.AsInteger[iPtlStatusCode] = 4) then
      acEditPatientData.Caption := 'Conf&irm...'
  else
    acEditPatientData.Caption := 'Ed&it...';

  updatePatientCounter;

  Handled := True;
end;

procedure TFormGenericRegistry.acShowRegistryUsersExecute(Sender: TObject);
begin
  ShowUserList(RegInfo);
end;

procedure TFormGenericRegistry.acSiteParametersExecute(Sender: TObject);
var
  frm: TFormSiteParameters;
begin
  inherited;
  frm := TFormSiteParameters.Create(RegInfo);
  try
    frm.Edit;
  finally
    frm.Free;
  end;
end;

procedure TFormGenericRegistry.ApplyOptions;
var
  ie: TCCRInterfaceElement;
begin
  inherited;
  ie := CCROptions.IE[ieReadOnly];
  //--- Log tree
  tvLogs.Color := ie.Color;
  tvLogs.Font.Color := ie.FontColor;
  //--- Log details
  with htmlDetails do
  begin
    DefBackground := ie.Color;
    DefFontColor  := ie.FontColor;
    DefFontSize   := ie.FontSize;
  end;
end;

procedure TFormGenericRegistry.FormActivate(Sender: TObject);
begin
  inherited;
  lsvPatientList.Invalidate;
end;

procedure TFormGenericRegistry.FormClose(Sender: TObject;
  var Action: TCloseAction);
begin
  Action := caFree;
end;

function TFormGenericRegistry.GetLayoutSectionName: String;
begin
  Result := ClassName;
end;

procedure TFormGenericRegistry.LoadLayout;
var
  section: String;
begin
  try
    vxsFormlayout.Open;
    section := GetLayoutSectionName;
    try
      lsvPatientList.LoadLayout(vxsFormlayout, section);
      lsvTaskList.LoadLayout(vxsFormlayout, section);
    finally
      vxsFormlayout.Close;
    end;
  except
  end;
end;

procedure TFormGenericRegistry.mnuGenericReportsClick(Sender: TObject);
begin
  if tsTaskManager.TabVisible then
    PageControl.ActivePage := tsTaskManager;
end;

procedure TFormGenericRegistry.mnuRegistryClick(Sender: TObject);
begin
  if tsRegistry.TabVisible then
    PageControl.ActivePage := tsRegistry;
end;

procedure TFormGenericRegistry.mnuReportsClick(Sender: TObject);
var
  code: Integer;
begin
  if Sender is TMenuItem then
  begin
    code := TMenuItem(Sender).Tag;
    if code > 0 then
      ScheduleReport(IntToStr(code))
    else
      ScheduleReport;
  end;
end;

procedure TFormGenericRegistry.PageControlChange(Sender: TObject);
begin
  Application.ProcessMessages;
  case PageControl.ActivePageIndex of
    0: LoadTaskList;
    1: Activity_LoadList;
    2: begin
        if not CCRScreenReaderActive then
          ActiveControl := csePatient
        //-- Section 508: Display a warning regarding disabled controls.
        else
          MessageDlg508('', RSC50801, mtWarning, [mbOK], 0);
       end;
  end;
  PageControl.HelpContext := PageControl.ActivePage.HelpContext;
end;

procedure TFormGenericRegistry.PageControlEnter(Sender: TObject);
begin
  inherited;
  //-- Section 508: Display a warning regarding disabled controls.
  if CCRScreenReaderActive and (PageControl.ActivePageIndex = 2) then
    MessageDlg508('', RSC50801, mtWarning, [mbOK], 0);
  PageControl.HelpContext := PageControl.ActivePage.HelpContext;
end;

procedure TFormGenericRegistry.SaveLayout;
var
  section: String;
begin
  try
    vxsFormLayout.Open;
    section := GetLayoutSectionName;
    try
      lsvPatientList.SaveLayout(vxsFormLayout, section);
      lsvTaskList.SaveLayout(vxsFormLayout, section);
    finally
      vxsFormLayout.Close;
    end;
  except
  end;
end;

procedure TFormGenericRegistry.setActivePatientData(Data: TPatientData);
begin
  try
    fActivePatientData.Free;
  except
  end;
  fActivePatientData := Data;
end;

procedure TFormGenericRegistry.SetupReportsMenu;
var
  i: Integer;
  mnu: TMenuItem;
  tsl: TStringList;
begin
  if not Assigned(RegInfo) then Exit;

  mnuGenericMain.BeginUpdate;
  try
    //--- Menu items for the reports
    tsl := TStringList.Create;
    try
      //--- Sort reports alphabetically
      tsl.Sorted := True;
      with RegInfo do
        for i:=AvailableReports.Count-1 downto 0 do
          tsl.AddObject(AvailableReports[i].Name, AvailableReports[i]);
      //--- Generate the menu items
      for i:=0 to tsl.Count-1 do
        with tsl.Objects[i] as TReportDescriptor do
        begin
          mnu := TMenuItem.Create(Self);
          mnu.Caption := Name;
          mnu.OnClick := mnuReportsClick;
          mnu.Tag     := StrToIntDef(Code, 0);
          mnuGenericReports.Add(mnu);
        end;
    finally
      tsl.Free;
    end;

    //--- Separator
    mnu := TMenuItem.Create(Self);
    mnu.Caption := '-';
    mnuGenericReports.Add(mnu);

    //--- Menu item that displays the report list
    mnu := TMenuItem.Create(Self);
    mnu.Caption     := 'Report List...';
    mnu.OnClick     := mnuReportsClick;
    mnu.Tag         := -1;
    mnu.HelpContext := 813;
    mnuGenericReports.Add(mnu);
  finally
    mnuGenericMain.EndUpdate;
  end;
end;

procedure TFormGenericRegistry.vxsFormLayoutBeforeOpen(Sender: TObject);
begin
  inherited;
  vxsFormLayout.Broker := Broker;
end;

{$REGION ' Task Manager Methods '}
///////////////////////////////// Task Manager \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

procedure TFormGenericRegistry.acAddTaskExecute(Sender: TObject);
begin
  ScheduleReport;
end;

procedure TFormGenericRegistry.acDeleteTaskExecute(Sender: TObject);
var
  taskItem: TCCRGridItem;
  taskNumber: String;
  pending, rc: Integer;
  msgRc: Word;
begin
    with lsvTaskList do
      begin
        pending := 0;
        msgRc := mrNo;
        while Assigned(Selected) do
          begin
            taskItem := Selected;
            taskItem.Selected := False;
            taskNumber := taskItem.AsString[iTskNumber];
            if taskNumber = '' then
              Continue;
              
            if msgRc <> mrYesToAll then
            begin
              msgRc := MessageDlg508(rscConfirmation, Format(RSC00141, [taskNumber]),
                mtConfirmation, [mbYes,mbNo,mbCancel,mbYesToAll], 0);
              if msgRc = mrCancel then
                Break;
            end;

            if (msgRc = mrYesToAll) or (msgRc = mrYes) then
              begin
                rc := Tasks_Delete(taskNumber);
                if rc = 0 then
                  taskItem.Free
                else if rc >0 then
                  Inc(pending);
              end;
          end;
        if pending > 0 then
        begin
          LoadTaskList;
          MessageDlg508(rscWarning, RSC00140, mtWarning, [mbOK], 0);
        end;
      end;
end;

procedure TFormGenericRegistry.acOpenReportExecute(Sender: TObject);
var
  code, task: String;
  report: TReportDescriptor;
begin
  if Assigned(lsvTaskList.Selected) then
    begin
      updateTaskList;
      if lsvTaskList.Selected.AsString[iTskCompleted] = '' then
      begin
        MessageDlg508(rscInformation, RSC00143, mtInformation, [mbOK], 0);
        Exit;
      end;
      task := lsvTaskList.Selected.AsString[iTskNumber];
      if task <> '' then
        begin
          code := lsvTaskList.Selected.AsString[iTskReportCode];
          report := RegInfo.AvailableReports.FindDescriptor(code);
          if Assigned(report) then
            TFormXMLReport.View(task, report.DefaultSorting)
          else
            TFormXMLReport.View(task);
        end;
    end;
end;

procedure TFormGenericRegistry.acRefreshTaskListExecute(Sender: TObject);
begin
  LoadTaskList;
end;

procedure TFormGenericRegistry.acViewTaskLogExecute(Sender: TObject);
begin
  if Assigned(lsvTaskList.Selected) then
    if not OpenLog(lsvTaskList.Selected.AsString[iTskLogIEN]) then
      MessageDlg508('', RSC00144, mtWarning, [mbOK], 0);
end;

procedure TFormGenericRegistry.LoadTaskList;
begin
  with lsvTaskList do
    begin
      Items.BeginUpdate;
      try
        Items.Clear;
        updateTaskList;
      finally
        Items.EndUpdate;
      end;
    end;
end;

procedure TFormGenericRegistry.lsvTaskListDblClick(Sender: TObject);
begin
  acOpenReportExecute(Sender);
end;

procedure TFormGenericRegistry.lsvTaskListKeyPress(Sender: TObject;
  var Key: Char);
begin
  inherited;
  if Key = #13 then
    acOpenReportExecute(Sender);
end;

function TFormGenericRegistry.ScheduleReport(aReportCode: String): TReturnCode;
var
  xmlParams: TStringList;
  {$IFDEF DBUGETM4}
  DebugETM: TextFile;
  DebugETMisTrue: boolean;
  {$ENDIF}
begin
  Result := RC_OK;
  if not Assigned(RegInfo) then
    Exit;

  {$IFDEF DBUGETM4}
  DebugETMisTrue := FileExists('C:\ETM\DEBUG-ETM-4.txt');
  if DebugETMisTrue then
  begin
    System.AssignFile(DebugETM, 'C:\ETM\DEBUG-ETM-4.txt');
    System.Append(DebugETM);
    writeln(DebugETM, '*********Start of ScheduleReport*********');
    writeln(DebugETM, 'aReportCode=['+aReportCode+']');
  end;
  {$ENDIF}

  with ReportSelector do
    begin
      if Open(RegInfo, aReportCode) and Assigned(ReportDescriptor) then
        try
          Broker.CreateResults;
          xmlParams := TStringList.Create;
          try
            ReportDescriptor.Parameters.AssignTo(xmlParams);
            {$IFDEF DBUGETM4}
            if DebugETMisTrue then
              writeln(DebugETM,
                'RPC:      <' + rpcReportSchedule     + '>'#13 +
                'Report:   <' + ReportDescriptor.Code + '>'#13 +
                'RegIEN:   <' + RegInfo.IEN           + '>'#13 +
                'Start at: <' + DateTimeToStart       + '>'#13 +
                'Schedule: <' + Schedule              + '>'#13#13 +
                'Params:'     + #13 +
                xmlParams.Text);
            {$ENDIF}
            if Broker.CallProc(rpcReportSchedule,
              [ReportDescriptor.Code, RegInfo.IEN, DateTimeToStart, Schedule],
              xmlParams) then
                LoadTaskList;
          finally
            xmlParams.Free;
          end;
        except
        end;
      {$IFDEF DBUGETM4}
      if DebugETMisTrue then
      begin
        writeln(DebugETM, '*********End of ScheduleReport*********');
        System.CloseFile(DebugETM);
      end;
      {$ENDIF}
    end;
end;

function TFormGenericRegistry.Tasks_Delete(ATask: String): Integer;
begin
  Result := -1;
  Broker.CreateResults;
  if Broker.CallProc(rpcTaskDelete, [ATask]) then
    Result := StrToIntDef(Piece(Broker.Results[0], U), 0);
end;

procedure TFormGenericRegistry.updateTaskList;
var
  item: TCCRGridItem;
  base, i, ntsk, tin: Integer;
  task, tmp: String;
begin
  Broker.CreateResults;
  with lsvTaskList do
    begin
      Items.BeginUpdate;
      try
        if Broker.CallProc(rpcTaskList, [RegInfo.IEN]) then
          begin
            tmp  := Broker.Results[0];
            ntsk := StrToIntDef(Piece(tmp, U, 1), 0);
            tin  := StrToIntDef(Piece(tmp, U, 2), 0);
            for i:=1 to ntsk do
              begin
                base := (i - 1) * tin;
                task := Broker.Results[base+1];

                item := TCCRGridItem(FindString(task, iTskNumber));
                if not Assigned(item) then
                  item := Items.Add;

                with Broker do
                  begin
                    //--- Dummy
                    item.AsString[iTskDummy] := ' ';
                    //--- Report code
                    item.AsString[iTskReportCode] := Piece(Results[base+4], U);
                    //--- Task number
                    item.AsString[iTskNumber] := task;
                    //--- Task type
                    item.AsString[iTskType] := Piece(Results[base+2], U, 2);
                    //--- Description
                    item.AsString[iTskDescription] := Results[base+5];
                    //--- Scheduled to run at
                    tmp := Piece(Results[base+11], U);
                    if tmp <> '' then item.AsString[iTskScheduled] := tmp;
                    //--- Task status
                    item.AsString[iTskStatus] := Piece(Results[base+6], U, 2);
                    //--- Progress percentage
                    item.AsString[iTskProgress] := Results[base+10];
                    //--- Completion time
                    tmp := Piece(Results[base+9], U);
                    if tmp <> '' then item.AsString[iTskCompleted] := tmp;
                    //--- Log IEN
                    item.AsString[iTskLogIEN] := Results[base+12];
                    //--- User comment
                    item.AsString[iTskComment] := Results[base+14];
                  end;
              end;
            SortField := SortField;
          end;
      finally
        Items.EndUpdate;
      end;
    end;
end;
{$ENDREGION}

{$REGION ' Technical Log Methods '}
///////////////////////////////// Technical Log \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

procedure TFormGenericRegistry.acMsgCopyExecute(Sender: TObject);
begin
  htmlDetails.CopyToClipboard;
end;

procedure TFormGenericRegistry.acMsgSelectAllExecute(Sender: TObject);
begin
  htmlDetails.SelectAll;
end;

procedure TFormGenericRegistry.acRefreshActivityLogExecute(Sender: TObject);
begin
  Activity_LoadList;
end;

procedure TFormGenericRegistry.Activity_LoadList;
var
  sFrom, sTo: String;

  procedure LoadAccessViolations(const sFrom, sTo: String);
  var
    i, n: Integer;
    node: TCCRAVNode;
  begin
    if Broker.CallProc(rpcLogGetViolations, [sFrom,sTo]) then
      begin
        n := StrToIntDef(Piece(Broker.Results[0],U), 0);
        node := TCCRAVNode.Create;
        try
          for i:=1 to n do
            node.List.Add(Broker.Results[i]);
          if node.List.Count > 0 then
            with tvLogs.Items.AddChildObject(nil, 'Access Violations', node) do
              begin
                ImageIndex    := node.GetImageIndex(itDefault);
                SelectedIndex := node.GetImageIndex(itSelected);
              end
          else
            FreeAndNil(node);
        except
          FreeAndNil(node);
          raise;
        end;
      end;
  end;

begin
  Broker.CreateResults;

  sFrom := FloatToStr(WindowsDateTimeToFMDateTime(dtpFrom.Date));
  sTo   := FloatToStr(WindowsDateTimeToFMDateTime(dtpTo.Date));

  Screen.Cursor := crHourGlass;
  try
    tvLogs.Items.BeginUpdate;
    try
      //--- Clear the tree and details
      tvLogs.Items.Clear;
      htmlDetails.Clear;
      //--- Load the list of acccess violations (if available)
      LoadAccessViolations(sFrom, sTo);
      //--- Load the list of logs
      if Broker.CallProc(rpcLogGetList, [RegInfo.IEN,sFrom,sTo]) then
        AddLogNodes(Broker.Results);
      //-- Section 508: Set up special handling for JAWS since it doesn't read
      //   the HTML correctly.
      if CCRScreenReaderActive then
      begin
        stxtLogDateFrom.TabStop := true;
        stxtLogDateTo.TabStop := true;
        htmlDetails.Visible   := false;
        pnlDetails508.Caption := ' ';
        pnlDetails508.Enabled := true;
        pnlDetails508.Visible := true;
        Clear508LogDetails;
        tvLogs.Selected := tvLogs.Items.GetFirstNode;
      end;
    finally
      tvLogs.Items.EndUpdate;
    end;
  finally
    Screen.Cursor := crDefault;
  end;
end;

function TFormGenericRegistry.AddLogNodes(RawData: TStrings): Integer;
var
  i, n: Integer;
  dts, s: String;
  node: TCCRLogNode;

  function AddLogNode(aLogNode: TCCRLogNode): TTreeNode;
  var
    s: String;
  begin
    s := Piece(Piece(aLogNode.Descr, U, 5), ':', 1, 2) + '  ' +
      Piece(aLogNode.Descr, U, 3);
    Result := tvLogs.Items.AddChildObject(nil, s, aLogNode);
    with Result do
      begin
        HasChildren   := StrToBoolDef(Piece(aLogNode.Descr, U, 6), False);
        ImageIndex    := aLogNode.GetImageIndex(itDefault);
        SelectedIndex := aLogNode.GetImageIndex(itSelected);
      end;
  end;

begin
  Result := 0;
  n := RawData.Count - 1;
  node := nil;
  for i:=1 to n do
    try
      s := RawData[i];
      dts := Piece(s, U);
      if dts = 'L' then
        begin
          if Piece(s, U, 2) <> 'END' then
            node := TCCRLogNode.Create(s)
          else if Assigned(node) then
            begin
              AddLogNode(node);
              Inc(Result);
              node := nil;
            end;
        end
      else if dts = 'R' then
        begin
          if Assigned(node) then
            node.RegLst.Add(s)
        end;
    except
      FreeAndNil(node);
      raise;
    end;
end;

procedure TFormGenericRegistry.LoadLogMessages(aNode: TTreeNode);
var
  i, n: Integer;
  dts, s: String;
  logNode: TCCRLogNode;
  node: TCCRMessageNode;

  function AddMsgNode(aMsgNode: TCCRMessageNode): TTreeNode;
  begin
    Result := tvLogs.Items.AddChildObject(aNode, aMsgNode.Msg, aMsgNode);
    with Result do
      begin
        ImageIndex    := aMsgNode.GetImageIndex(itDefault);
        SelectedIndex := aMsgNode.GetImageIndex(itSelected);
      end;
  end;

begin
  Broker.CreateResults;

  logNode := TCCRLogNode(aNode.Data);

  if Broker.CallProc(rpcLogGetMessages, [logNode.IEN]) then
    begin
      n := Broker.Results.Count - 1;
      tvLogs.Items.BeginUpdate;
      try
        node := nil;
        for i:=1 to n do
          try
            s := Broker.Results[i];
            dts := Piece(s, U);
            if dts = 'M' then
              begin
                if Piece(s, U, 2) <> 'END' then
                  node := TCCRMessageNode.Create(s)
                else
                  begin
                    AddMsgNode(node);
                    node := nil;
                  end
              end
            else if dts = 'A' then
              begin
                if Assigned(node) then
                  node.Details.Add(s)
              end
            else if dts = 'T' then
              begin
                if Assigned(node) then
                  node.Msg := Piece(s, U, 2, 999);
              end
          except
            FreeAndNil(node);
            raise;
          end;
      finally
        tvLogs.Items.EndUpdate;
      end;
    end;
end;

procedure TFormGenericRegistry.mnuLogsPopup(Sender: TObject);
begin
  acMsgCopy.Enabled := (htmlDetails.SelLength > 0);
end;

function TFormGenericRegistry.OpenLog(const aLogIEN: String): Boolean;
var
  tn: TTreeNode;

  function findLogNode(const aLogIEN: String): TTReeNode;
  var
    tn: TTreeNode;
    node: TCCRActivityNode;
  begin
    Result := nil;
    tn := tvLogs.Items.GetFirstNode;
    while Assigned(tn) do
      begin
        node := TCCRActivityNode(tn.Data);
        if node is TCCRLogNode then
          if TCCRLogNode(node).IEN = aLogIEN then
            begin
              Result := tn;
              Break;
            end;
        tn := tn.getNextSibling;
      end;
  end;

begin
  Result := False;
  Broker.CreateResults;
  //--- Search for the log in the list
  tn := findLogNode(aLogIEN);
  //--- Try to add the log to the list
  if not Assigned(tn) then
    if Broker.CallProc(rpcLogInfo, [aLogIEN]) then
      if StrToIntDef(Piece(Broker.Results[0],U), 0) > 0 then
        begin
          tvLogs.Items.Clear;
          AddLogNodes(Broker.Results);
          tn := tvLogs.Items.GetFirstNode;
        end;
  //--- Open the log
  if Assigned(tn) then
    begin
      tvLogs.FullCollapse;
      tvLogs.Selected := tn;
      tn.Expand(False);
      PageControl.ActivePage := tsActivityLog;
      Result := True;
    end;
end;

//-- Section 508: Clear all 508 static text controls
procedure TFormGenericRegistry.Clear508LogDetails;
var i: Integer;
begin
  for i := 0 to pnlDetails508.ControlCount-1 do
  begin
    if pnlDetails508.Controls[i] is TStaticText then
    begin
      TStaticText(pnlDetails508.Controls[i]).TabStop := false;
      TStaticText(pnlDetails508.Controls[i]).Visible := false;
    end;
  end;
  stxtTitle.TabStop := true;
  stxtTitle.Visible := true;
  stxtTitle.Caption := 'No data to display';
end;

//-- Section 508: Load all 508 static text controls with node data
procedure TFormGenericRegistry.Show508LogDetails(const aLogDetail: String);
var i,j,n: Integer;
s: String;
begin
  // Get the number of pieces of data
  n := (Length(aLogDetail) - Length(StringReplace(aLogDetail, U, '',[rfReplaceAll])))
    div Length(U);
  for i := 1 to n do
  begin
    if (i <= pnlDetails508.ControlCount) and (pnlDetails508.Controls[i-1] is TStaticText) then
    begin
      s := Piece(aLogDetail,U,i);
      TStaticText(pnlDetails508.Controls[i-1]).Caption := s;
      TStaticText(pnlDetails508.Controls[i-1]).TabStop := true;
      TStaticText(pnlDetails508.Controls[i-1]).Visible := true;
      j := Length(s) - Length(StringReplace(s, chr(13), '',[rfReplaceAll]));
      if j > 0 then
        TStaticText(pnlDetails508.Controls[i-1]).Height := (j+1)*17;
    end;
  end;
end;

procedure TFormGenericRegistry.tsActivityLogEnter(Sender: TObject);
begin
  inherited;
  if CCRScreenReaderActive then
    MessageDlg508('', Format(RSC50802, ['Technical Log']), mtInformation, [mbOK], 0);
end;

procedure TFormGenericRegistry.tsTaskManagerEnter(Sender: TObject);
begin
  inherited;
  if CCRScreenReaderActive then
    MessageDlg508('', Format(RSC50802, ['Task Manager']), mtInformation, [mbOK], 0);
end;

procedure TFormGenericRegistry.tvLogsChange(Sender: TObject;
  Node: TTreeNode);
var
  activityNode: TCCRActivityNode;
begin
  activityNode := TCCRActivityNode(Node.Data);
  //-- Section 508: Set up special handling for JAWS since it doesn't read the
  //   the HTML correctly.
  if CCRScreenReaderActive then
  begin
    Clear508LogDetails;
    if Assigned(activityNode) then
      Show508LogDetails(activityNode.Get508Details);
  end
  else begin
    if Assigned(activityNode) then
      htmlDetails.LoadFromString(activityNode.GetHTML(HTMLParams), '')
    else
      htmlDetails.Clear;
  end;
end;

procedure TFormGenericRegistry.tvLogsKeyPress(Sender: TObject; var Key: Char);
begin
  inherited;
  if (Key = #13) and (tvLogs.Selected.Level = 0) then
  begin
    if (tvLogs.Selected.Expanded) then
      tvLogsCollapsed(tvLogs, tvLogs.Selected)
    else
      tvLogs.Selected.Expand(true);
  end;
end;

procedure TFormGenericRegistry.tvLogsCollapsed(Sender: TObject;
  Node: TTreeNode);
var
  activityNode: TCCRActivityNode;
begin
  with Node do
    begin
      activityNode := TCCRActivityNode(Node.Data);
      if Assigned(activityNode) then
        begin
          Node.ImageIndex    := activityNode.GetImageIndex(itDefault);
          Node.SelectedIndex := activityNode.GetImageIndex(itSelected);
        end;
      DeleteChildren;
      HasChildren := True;
    end;
end;

procedure TFormGenericRegistry.tvLogsDeletion(Sender: TObject;
  Node: TTreeNode);
begin
  if Assigned(Node.Data) then
    begin
      TObject(Node.Data).Free;
      Node.Data := nil;
    end;
end;

procedure TFormGenericRegistry.tvLogsExpanding(Sender: TObject;
  Node: TTreeNode; var AllowExpansion: Boolean);
var
  activityNode: TCCRActivityNode;
begin
  if Node.Level = 0 then
    begin
      LoadLogMessages(Node);
      if Node.HasChildren then
        begin
          activityNode := TCCRActivityNode(Node.Data);
          if Assigned(activityNode) then
            begin
              Node.ImageIndex    := activityNode.GetImageIndex(itExpanded);
              Node.SelectedIndex := Node.ImageIndex;
            end
        end
      else
        AllowExpansion := False;
    end;
end;
{$ENDREGION}

/////////////////////////////////// Registry \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

procedure TFormGenericRegistry.acDeletePatientExecute(Sender: TObject);
var
  s: String;
begin
  if not SetPatientContext then
    Exit;

  StartPatientEdit;
  try
    s := Format(RSC00122, [lsvPatientList.ItemFocused.AsString[iPtlName], PtIEN]);
    if MessageDlg508(RSC00125, s, mtConfirmation, [mbYes, mbNo], 0) = mrYes then
      begin
        Broker.CreateResults;
        if Broker.CallProc(rpcPatientDelete, [RegInfo.IEN, PtIEN]) then
          begin
            if Broker.Results[0] = '0' then
              MessageDlg508('', RSC00123, mtInformation, [mbOK], 0)
            else
              lsvPatientList.Items.Delete(lsvPatientList.ItemFocused.Index);
          end
        else
          MessageDlg508('', RSC00124, mtInformation, [mbOK], 0);
      end;
  finally
    EndPatientEdit;
  end;
end;

procedure TFormGenericRegistry.acEditPatientDataExecute(Sender: TObject);
begin
  ReloadActivePatientRecord(PtIEN);
end;

procedure TFormGenericRegistry.btnPatientSearchClick(Sender: TObject);
begin
  csePatient.Search;
end;

procedure TFormGenericRegistry.cbClearScreen(Sender: TObject);
begin
  if Sender is TCheckbox then
    if TCheckbox(Sender).Checked then
    begin
      cbScreen.Checked := False;
      lsvPatientList.Fields[iPtlComment].Visible := cbPending.Visible;
    end;
end;

procedure TFormGenericRegistry.cbScreenClick(Sender: TObject);
begin
  dtpScreenDate.Color   := CtrlColor(cbScreen.Checked);
  dtpScreenDate.Enabled := cbScreen.Checked;
  if cbScreen.Checked then
  begin
    cbPending.Checked := False;
    lsvPatientList.Fields[iPtlComment].Visible := False;
  end
  else if cbPending.Visible then
    lsvPatientList.Fields[iPtlComment].Visible := True;
end;

procedure TFormGenericRegistry.CCOWResumed;
begin
  inherited;
  lsvPatientList.Invalidate;
end;

procedure TFormGenericRegistry.CCOWSuspended;
begin
  inherited;
  lsvPatientList.Invalidate;
end;

procedure TFormGenericRegistry.csePatientSearch(aSender: TObject;
  var aCommand: TCCRSearchCommand);

  procedure FillPatientList;
  var
    sTmp: String;
    i: Integer;
    LItem: TCCRGridItem;
  begin
    i := 1;
    with csePatient do
      try
        while i < RawData.Count-1 do
          begin
            if SearchGrid.Items.Count >= SearchMaxCount then
              Break;

            LItem := SearchGrid.Items.Add;
            LItem.ImageIndex := -1;
            LItem.AsString[iPtlName] := Piece(RawData[i], U, 2);
            with LItem do
              begin
                sTmp := RawData[i];
                AsString[iPtlDummy] := ' ';
                AsString[iPtlIEN]  := Piece(sTmp, U, 1);
                AsString[iPtlName] := Piece(sTmp, U, 2);
                AsString[iPtlDOB]  := Piece(sTmp, U, 3);
                AsString[iPtlSSN]  := FormatSSN(Piece(sTmp, U, 4));
                AsString[iPtlDOD]  := Piece(Piece(sTmp, U, 5), '.');
                AsString[iPtlSex]  := Piece(sTmp, U, 6);

                sTmp := RawData[i+1];

                AsString[iPtlStatusCode]    := Piece(sTmp, U,  3);
                AsString[iPtlStatus]        := PatientStatusText(Piece(sTmp,U,3));
                AsString[iPtlAcknowledged]  := Piece(sTmp, U,  6);
                AsString[iPtlExtracted]     := Piece(sTmp, U,  7);
                AsString[iPtlSelected]      := Piece(sTmp, U,  8);
                AsString[iPtlConfirmed]     := Piece(sTmp, U,  9);
                AsString[iPtlSite]          := Piece(sTmp, U, 10);
                AsString[iPtlSelectionRule] := Piece(sTmp, U, 11);
                AsString[iPtlFlags]         := Piece(sTmp, U, 14);
                AsString[iPtlComment]       := Piece(sTmp, U, 15);
              end;
            i := i + 2;
          end;
        except
        end;
  end;

var
  n: Integer;
begin
  with csePatient do
    if Broker.CallProc(rpcListPatients, SearchParams, nil, RawData) then
      begin
        n := StrToIntDef(Piece(RawData[0],U), 0);
        if n > 0 then
          begin
            Inc(SearchCount, n);
            SearchParams[SearchFromIndex] := Piece(RawData[0], U, 2, 99);

            //--- Add the records to the list
            FillPatientList;

            //--- Check if there are more records to load
            if SearchParams[SearchFromIndex] = '' then
              aCommand := cscFinish
            //--- Check if the maximum number of records has been reached
            else if (SearchMaxCount > 0) and (SearchCount > SearchMaxCount) then
              aCommand := cscFinish;

            updatePatientCounter;
          end
        else
          aCommand := cscFinish;
      end
    else
      aCommand := cscCancel;
end;

procedure TFormGenericRegistry.csePatientSearchEnd(aSender: TObject;
  var aCommand: TCCRSearchCommand);

  procedure FocusPatient;
  begin
    with csePatient do
      if SearchGrid.Items.Count > 0 then
        try
          SearchGrid.ItemFocused := SearchGrid.Items[0];
          SearchGrid.ItemFocused.Selected := True;
          Self.ActiveControl := SearchGrid;
        except
        end;
  end;

begin
  DisplayStatus('');

  with csePatient do
    if SearchCount > 0 then
      begin
        if SearchCount > SearchMaxCount then
        begin
          Application.ProcessMessages;
          MessageDlg508(rscPatientSelection, RSC00126, mtInformation, [mbOK], 0);
        end
        else begin
          if aCommand = cscCancel then
            MessageDlg508(rscPatientSelection, RSC00128, mtWarning, [mbOK], 0);
          SearchGrid.AlphaSort;
        end;
      end
    else if (fPatientIEN = '') then
      if aCommand = cscCancel then
        MessageDlg508(rscPatientSelection, RSC00128, mtWarning, [mbOK], 0)
      else
        MessageDlg508(rscPatientSelection, RSC00127, mtInformation, [mbOK], 0);

  //--- Set the focus to the patient count if a screen reader is being used.
  if stxtCount.TabStop then
    stxtCount.SetFocus
  else
    FocusPatient;

  //--- Re-enable the Search button after the search is complete
  btnPatientSearch.Enabled := True;
end;

procedure TFormGenericRegistry.csePatientSearchStart(
  Sender: TObject; var aCommand: TCCRSearchCommand);
var
  sDate, sFlags, sTarget: String;
begin
  //--- Disable the Search button until the search is complete
  btnPatientSearch.Enabled := False;

  //--- Prepare parameters
  if fPatientIEN <> '' then
    begin
      sDate   := '';
      sTarget := '`' + fPatientIEN;
      sFlags  := 'O';
    end
  else
    begin
      sFlags := 'O';
      if cbScreen.Checked then
        begin
          sDate := FloatToStr(WindowsDateToFMDate(Trunc(dtpScreenDate.Date)));
          sFlags := sFlags + 'C';
        end
      else
        sDate := '';
      if cbPending.Checked then
        sFlags := sFlags + 'P';
      sTarget := Uppercase(csePatient.Text);
    end;

  //--- Update the interface
  updatePatientCounter;
  DisplayStatus(RSC00129);
  Application.ProcessMessages;

  //--- Store the search parameters
  with csePatient do
    begin
      SetParams([RegInfo.IEN, sDate, sTarget, sFlags, '20', '']);
      SearchMaxCount := CCROptions.MaxPatientCount;
    end;
end;

procedure TFormGenericRegistry.LoadPatientList(aPatientIEN: String);
begin
  fPatientIEN := aPatientIEN;
  try
    csePatient.Search;
  finally
    fPatientIEN := '';
  end;
end;

procedure TFormGenericRegistry.lsvPatientListCustomDrawItem(
  Sender: TCustomListView; Item: TListItem; State: TCustomDrawState;
  var DefaultDraw: Boolean);
begin
  if (TCCRGridItem(Item).AsString[iPtlIEN] = CCOWContextor.PatientDFN)
    and (CCOWContextor.State = csParticipating) then
    Item.ImageIndex := 2
  else
    Item.ImageIndex := -1;
end;

procedure TFormGenericRegistry.lsvPatientListDblClick(Sender: TObject);
begin
  if acEditPatientData.Enabled then
    acEditPatientData.Execute;
end;

procedure TFormGenericRegistry.lsvPatientListInsert(Sender: TObject;
  Item: TListItem);
begin
  Item.Indent := 0;
end;

procedure TFormGenericRegistry.lsvPatientListKeyDown(Sender: TObject;
  var Key: Word; Shift: TShiftState);
begin
  inherited;
  if (Key = VK_RETURN) and acEditPatientData.Enabled then
    acEditPatientData.Execute;
end;

procedure TFormGenericRegistry.ReloadActivePatientRecord(const aPatientIEN: String);
begin
  if not IsRCError(ActivePatientData.Load(aPatientIEN, False)) then
    with ActivePatientData do
      with lsvPatientList.ItemFocused do
        begin
          AsString[iPtlAcknowledged]     := AcknowledgedUntil;
          AsString[iPtlDOB]              := DOB;
          AsString[iPtlDOD]              := DOD;
          AsString[iPtlConfirmed]        := DateConfirmed;
          AsString[iPtlExtracted]        := ExtractedUntil;
          AsString[iPtlFlags]            := Flags;
          AsString[iPtlIEN]              := PatientIEN;
          AsString[iPtlName]             := PatientName;
          AsString[iPtlSex]              := Sex;
          AsString[iPtlSSN]              := FormatSSN(SSN);
          AsString[iPtlStatusCode]       := Status;
          AsString[iPtlStatus]           := PatientStatusText(Status);
          AsString[iPtlSelected]         := DateSelected;
          AsString[iPtlSite]             := EarliestRuleLoc;
          AsString[iPtlSelectionRule]    := EarliestRuleDescr;
          AsString[iPtlComment]          := PendingComment;

          SwitchActivePatient(lsvPatientList.ItemFocused);
        end;
end;

procedure TFormGenericRegistry.SelectPatient(aPatientIEN: String);
var
  i, n: Integer;
  found: Boolean;
begin
  with tsRegistry do
    if not (Visible and Enabled) then
      Exit;

  if aPatientIEN <> '' then
    begin
      found  := False;
      with lsvPatientList do
        begin
          n := Items.Count - 1;
          for i:=0 to n do
            if Items[i].AsString[iPtlIEN] = aPatientIEN then
              begin
                if Assigned(ItemFocused) then
                  ItemFocused.Update;
                Selected := Items[i];
                ItemFocused := Selected;
                Selected.MakeVisible(False);
                SwitchActivePatient(Selected);
                found := True;
                Break;
              end;
        end;
      if not found then
        LoadPatientList(aPatientIEN);
    end
  else
    lsvPatientList.Clear;

  inherited;
end;

function TFormGenericRegistry.SetPatientContext: Boolean;
begin
  if CCOWContextor.SetPatientContext(PtIEN) then
    begin
      Result := True;
      lsvPatientList.Invalidate;
    end
  else
    Result := False;
end;

procedure TFormGenericRegistry.SwitchActivePatient(anItem: TCCRGridItem);
var
  flags: String;
begin
  if Assigned(anItem) then
    begin
      flags := anItem.AsString[iPtlFlags];
      acDeletePatient.Enabled   := Pos('D', flags) <> 0;
      acEditPatientData.Enabled := Pos('E', flags) <> 0;
      PtIEN                     := anItem.AsString[iPtlIEN];
    end
  else
    begin
      acDeletePatient.Enabled   := False;
      acEditPatientData.Enabled := False;
      PtIEN                     := '';
    end;
end;

procedure TFormGenericRegistry.updatePatientCounter;
begin
  stxtCount.Caption := Format('Patients found: %d', [lsvPatientList.Items.Count]);
end;

procedure TFormGenericRegistry.pnlToolbarTasksResize(Sender: TObject);
begin
  with btnDeleteTask do
    Left := pnlToolbarTasks.Width - Width - 1;
  with btnTasksSpacer do
    Width := btnDeleteTask.Left - Left;
end;

procedure TFormGenericRegistry.pnlToolbarLogsResize(Sender: TObject);
begin
  with pnlLogsSpacer do
    Width := pnlToolbarLogs.Width - Left - 1;
end;

procedure TFormGenericRegistry.pnlToolbarRegistryResize(Sender: TObject);
begin
  with btnPatientDelete do
    Left := pnlToolbarRegistry.Width - Width - 1;
  with btnRegistrySpacer do
    Width := btnPatientDelete.Left - Left;
end;

end.

